/*
  Arduino.h - Main include file for the Arduino SDK
  Copyright (c) 2005-2013 Arduino Team.  All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef Arduino_h
#define Arduino_h

#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <math.h>

#include <avr/pgmspace.h>
#include <avr/io.h>
#include <avr/interrupt.h>

#include "binary.h"

#ifdef __cplusplus
extern "C"{
#endif

void yield(void);

#define HIGH 0x1
#define LOW  0x0

#define INPUT 0x0
#define OUTPUT 0x1
#define INPUT_PULLUP 0x2
#define ANALOG	0x3

#define PI 3.1415926535897932384626433832795
#define HALF_PI 1.5707963267948966192313216916398
#define TWO_PI 6.283185307179586476925286766559
#define DEG_TO_RAD 0.017453292519943295769236907684886
#define RAD_TO_DEG 57.295779513082320876798154814105
#define EULER 2.718281828459045235360287471352

#define SERIAL  0x0
#define DISPLAY 0x1

#define LSBFIRST 0
#define MSBFIRST 1

#define CHANGE 1
#define FALLING 2
#define RISING 3

#if defined(__AVR_ATtiny24__) || defined(__AVR_ATtiny44__) || defined(__AVR_ATtiny84__)
  #define DEFAULT 0
  #define EXTERNAL 1
  #define INTERNAL1V1 2
  #define INTERNAL INTERNAL1V1
#elif defined(__AVR_ATtiny25__) || defined(__AVR_ATtiny45__) || defined(__AVR_ATtiny85__)
  #define DEFAULT 0
  #define EXTERNAL 4
  #define INTERNAL1V1 8
  #define INTERNAL INTERNAL1V1
  #define INTERNAL2V56 9
  #define INTERNAL2V56_EXTCAP 13
#else  
#if defined(__AVR_ATmega1280__) || defined(__AVR_ATmega2560__) || defined(__AVR_ATmega1284__) || defined(__AVR_ATmega1284P__) || defined(__AVR_ATmega644__) || defined(__AVR_ATmega644A__) || defined(__AVR_ATmega644P__) || defined(__AVR_ATmega644PA__)
#define INTERNAL1V1 2
#define INTERNAL2V56 3
#else
#define INTERNAL 3
#define INTERNAL1V25 3
#define INTERNAL2V56 2
#define INTERNAL1V024 3
#define INTERNAL2V048 2
#define INTERNAL4V096 4
#endif
#define DEFAULT 1
#define EXTERNAL 0
#endif

// undefine stdlib's abs if encountered
#ifdef abs
#undef abs
#endif

#define min(a,b) ((a)<(b)?(a):(b))
#define max(a,b) ((a)>(b)?(a):(b))
#define abs(x) ((x)>0?(x):-(x))
#define constrain(amt,low,high) ((amt)<(low)?(low):((amt)>(high)?(high):(amt)))
#define round(x)     ((x)>=0?(long)((x)+0.5):(long)((x)-0.5))
#define radians(deg) ((deg)*DEG_TO_RAD)
#define degrees(rad) ((rad)*RAD_TO_DEG)
#define sq(x) ((x)*(x))

#define interrupts() sei()
#define noInterrupts() cli()

#define clockCyclesPerMicrosecond() ( F_CPU / 1000000L )
#define clockCyclesToMicroseconds(a) ( (a) / clockCyclesPerMicrosecond() )
#define microsecondsToClockCycles(a) ( (a) * clockCyclesPerMicrosecond() )

#define lowByte(w) ((uint8_t) ((w) & 0xff))
#define highByte(w) ((uint8_t) ((w) >> 8))

#define bitRead(value, bit) (((value) >> (bit)) & 0x01)
#define bitSet(value, bit) ((value) |= (1UL << (bit)))
#define bitClear(value, bit) ((value) &= ~(1UL << (bit)))
#define bitWrite(value, bit, bitvalue) (bitvalue ? bitSet(value, bit) : bitClear(value, bit))

// avr-libc defines _NOP() since 1.6.2
#ifndef _NOP
#define _NOP() do { __asm__ volatile ("nop"); } while (0)
#endif

typedef unsigned int word;

#define bit(b) (1UL << (b))

typedef bool boolean;
typedef uint8_t byte;

void init(void);
void initVariant(void);

int atexit(void (*func)()) __attribute__((weak));

void pinMode(uint8_t, uint8_t);
void digitalWrite(uint8_t, uint8_t);
int digitalRead(uint8_t);
void digitalToggle(uint8_t);
int analogRead(uint8_t);
#define ADC10BIT 10
#define ADC12BIT 12
void analogReference(uint8_t mode);
void analogWrite(uint8_t, uint16_t);
void analogReadResolution(uint8_t);

void pwmWrite(uint8_t, uint16_t);
void pwmTurnOff(uint8_t);

unsigned long millis(void);
unsigned long micros(void);
void delay(unsigned long);
void delayMicroseconds(unsigned int us);
unsigned long pulseIn(uint8_t pin, uint8_t state, unsigned long timeout);
unsigned long pulseInLong(uint8_t pin, uint8_t state, unsigned long timeout);

void shiftOut(uint8_t dataPin, uint8_t clockPin, uint8_t bitOrder, uint8_t val);
uint8_t shiftIn(uint8_t dataPin, uint8_t clockPin, uint8_t bitOrder);

void attachInterrupt(uint8_t, void (*)(void), int mode);
void detachInterrupt(uint8_t);

void setup(void);
void loop(void);

// Get the bit location within the hardware port of the given virtual pin.
// This comes from the pins_*.c file for the active board configuration.

#define analogInPinToBit(P) (P)

// On the ATmega1280, the addresses of some of the port registers are
// greater than 255, so we can't store them in uint8_t's.
extern const uint16_t PROGMEM port_to_mode_PGM[];
extern const uint16_t PROGMEM port_to_input_PGM[];
extern const uint16_t PROGMEM port_to_output_PGM[];

extern const uint8_t PROGMEM digital_pin_to_port_PGM[];
// extern const uint8_t PROGMEM digital_pin_to_bit_PGM[];
extern const uint8_t PROGMEM digital_pin_to_bit_mask_PGM[];
extern const uint8_t PROGMEM digital_pin_to_timer_PGM[];

// Get the bit location within the hardware port of the given virtual pin.
// This comes from the pins_*.c file for the active board configuration.
// 
// These perform slightly better as macros compared to inline functions
//
#define digitalPinToPort(P) ( pgm_read_byte( digital_pin_to_port_PGM + (P) ) )
#define digitalPinToBitMask(P) ( pgm_read_byte( digital_pin_to_bit_mask_PGM + (P) ) )
#define digitalPinToTimer(P) ( pgm_read_byte( digital_pin_to_timer_PGM + (P) ) )
#define analogInPinToBit(P) (P)
#define portOutputRegister(P) ( (volatile uint8_t *)( pgm_read_word( port_to_output_PGM + (P))) )
#define portInputRegister(P) ( (volatile uint8_t *)( pgm_read_word( port_to_input_PGM + (P))) )
#define portModeRegister(P) ( (volatile uint8_t *)( pgm_read_word( port_to_mode_PGM + (P))) )

#define NOT_A_PIN 0
#define NOT_A_PORT 0

#define NOT_AN_INTERRUPT -1

#ifdef ARDUINO_MAIN
#define PA 1
#define PB 2
#define PC 3
#define PD 4
#define PE 5
#define PF 6
#define PG 7
#define PH 8
#define PJ 10
#define PK 11
#define PL 12
#endif

#define NOT_ON_TIMER 0x00
#define TIMER0    0x10
#define TIMER0A   0x10
#define TIMER0B   0x11
#define TIMER0AX  0x12
#define TIMER0BX  0x13

#define TIMER1    0x20
#define TIMER1A   0x20
#define TIMER1B   0x21
#define TIMER1C   0x22
#define TIMER1AX  0x23
#define TIMER1BX  0x24

#define TIMER2    0x30
#define TIMER2A   0x31
#define TIMER2B   0x32
#define TIMER2AX  0x33
#define TIMER2BX  0x34

#define TIMER3    0x40
#define TIMER3A   0x40
#define TIMER3B   0x41
#define TIMER3C   0x42
#define TIMER3AX  0x43
#define TIMER3BX  0x44
#define TIMER3AA  0x45

#define TIMER4    0x50
#define TIMER4A   0x50
#define TIMER4B   0x51
#define TIMER4C   0x52
#define TIMER4D   0x53

#define TIMER5    0x60
#define TIMER5A   0x60
#define TIMER5B   0x61
#define TIMER5C   0x62

#define LGTDAO0   0xf0
#define LGTDAO1   0xf1

void unlockWrite(volatile uint8_t *, uint8_t);
void atomicWriteWord(volatile uint8_t *, uint16_t);

#ifdef __cplusplus
} // extern "C"
#endif

#ifdef __cplusplus
#include "WCharacter.h"
#include "WString.h"
#include "HardwareSerial.h"
#include "USBAPI.h"
#if defined(HAVE_HWSERIAL0) && defined(HAVE_CDCSERIAL)
#error "Targets with both UART0 and CDC serial not supported"
#endif

uint16_t makeWord(uint16_t w);
uint16_t makeWord(byte h, byte l);

#define word(...) makeWord(__VA_ARGS__)

unsigned long pulseIn(uint8_t pin, uint8_t state, unsigned long timeout = 1000000L);
unsigned long pulseInLong(uint8_t pin, uint8_t state, unsigned long timeout = 1000000L);

void tone(uint8_t _pin, unsigned int frequency, unsigned long duration = 0);
void noTone(uint8_t _pin);

// WMath prototypes
long random(long);
long random(long, long);
void randomSeed(unsigned long);
long map(long, long, long, long, long);
#endif

#include "pins_arduino.h"

#if defined(__LGT8FX8E__) || defined(__LGT8FX8P__)
#include "fastio_digital.h"

#define	INT_OSC_32K	0
#define	INT_OSC_32M	1
#define	EXT_OSC_32M	2
#define	EXT_OSC_24M	3
#define	EXT_OSC_16M	4
#define	EXT_OSC_12M	5
#define	EXT_OSC_8M	6
#define	EXT_OSC_4M	7
#define	EXT_OSC_2M	8
#define	EXT_OSC_1M	9
#define	EXT_OSC_400K 10
#define	EXT_OSC_32K 11

#define SYSCLK_DIV_0    0x0
#define SYSCLK_DIV_2    0x1
#define SYSCLK_DIV_4    0x2
#define SYSCLK_DIV_8    0x3
#define SYSCLK_DIV_16   0x4
#define SYSCLK_DIV_32   0x5
#define SYSCLK_DIV_64   0x6
#define SYSCLK_DIV_128  0x7

void sysClock(uint8_t mode);
void sysClockPrescale(uint8_t div);
void sysClockOutput(uint8_t enable);

#define chip_uuid()  ((GUID3 << 24) | (GUID2 << 16) | (GUID1 << 8) | GUID0)
// Log(HSP v3.7):
//  - for system tick based on timer 2
#if defined(TIMSK) && defined(TOIE2)
#define stopTick() do { TIMSK &= ~_BV(TOIE2); } while(0)
#elif defined(TIMSK2) && defined(TOIE2)
#define stopTick() do { TIMSK2 &= ~_BV(TOIE2); } while(0)
#else
#error  Timer2 overflow interrupt is not defined! 
#endif

#if defined(TIMSK) && defined(TOIE2)
#define startTick() do { TIMSK |= _BV(TOIE2); } while(0)
#elif defined(TIMSK2) && defined(TOIE2)
#define startTick() do { TIMSK |= _BV(TOIE2); } while(0)
#else
#error  Timer2 overflow interrupt is not defined! 
#endif
// Log(HSP v3.7): END

#endif

#ifndef nop
#define	nop()	__asm__ __volatile__ ("nop" ::)
#endif

#endif
